document.addEventListener('DOMContentLoaded', function() {
  // Spotifyカバー画像の初期設定
  const spotifyCover = document.getElementById('spotifyCover');
  if (spotifyCover) {
    spotifyCover.src = chrome.runtime.getURL('images/music.png');
  }
  
  // URLハッシュフラグメントをチェックして該当タブを開く
  const hash = window.location.hash.substring(1); // #を除く
  if (hash === 'settings') {
    setTimeout(() => {
      const settingsTab = document.querySelector('[data-tab="settings"]');
      if (settingsTab) {
        settingsTab.click();
        // Spotify設定セクションまでスクロール
        setTimeout(() => {
          scrollToSpotifySettings();
        }, 100);
      }
    }, 100);
  }
  
  // 設定を読み込んで初期値を設定
  loadSettings();
  
  // コース履歴を読み込む
  loadCourseHistory();
  
  // アカウント状態を初期化
  initializeAccount();
  
  // ログイン状態をチェックしてタイマーUIを初期化
  checkLoginStatusForTimer();

  // タイマーモード選択
  const timerModeButtons = document.querySelectorAll('.timer-mode-btn');
  let timerMode = 'manual'; // default

  // スキップボタン
  const skipToBreakBtn = document.getElementById('skipToBreak');
  const skipBreakBtn = document.getElementById('skipBreak');

  // タイマーモード保存・復元
  chrome.storage.sync.get('timerMode', (result) => {
    if (result.timerMode) {
      timerMode = result.timerMode;
      setActiveTimerMode(timerMode);
    }
    updateTimerModeUI();
  });

  function setActiveTimerMode(mode) {
    timerModeButtons.forEach(btn => {
      btn.classList.remove('active');
      if (btn.dataset.mode === mode) {
        btn.classList.add('active');
      }
    });
  }

  timerModeButtons.forEach(button => {
    button.addEventListener('click', function() {
      timerMode = this.dataset.mode;
      setActiveTimerMode(timerMode);
      chrome.storage.sync.set({ timerMode });
      updateTimerModeUI();
    });
  });

  function updateTimerModeUI() {
    const timerCircle = document.getElementById('timerCircle');
    const simpleTimer = document.getElementById('simpleTimer');
    
    if (timerMode === 'manual') {
      // 手動モードの場合はシンプルタイマーを表示
      timerCircle.style.display = 'none';
      simpleTimer.style.display = 'block';
      skipToBreakBtn.style.display = 'none';
      skipBreakBtn.style.display = 'none';
    } else {
      // ポモドーロ・45-15モードの場合は円形タイマーを表示
      timerCircle.style.display = 'block';
      simpleTimer.style.display = 'none';
    }
    // 現在のタイマー状態をリフレッシュ
    fetchStatus();
  }

  // スキップボタンイベント
  skipToBreakBtn.addEventListener('click', function() {
    chrome.runtime.sendMessage({ type: 'advancedTimerSkipToBreak' });
  });
  skipBreakBtn.addEventListener('click', function() {
    chrome.runtime.sendMessage({ type: 'advancedTimerSkipBreak' });
  });

  // タブ切り替え機能の実装
  const tabButtons = document.querySelectorAll('.tab-button');
  const tabContents = document.querySelectorAll('.tab-content');
  
  tabButtons.forEach(button => {
    button.addEventListener('click', () => {
      const tabId = button.getAttribute('data-tab');
      
      // タブボタンのアクティブ状態を切り替え
      tabButtons.forEach(btn => btn.classList.remove('active'));
      button.classList.add('active');
      
      // タブコンテンツの表示を切り替え
      tabContents.forEach(content => {
        content.classList.remove('active');
        if (content.id === tabId) {
          content.classList.add('active');
        }
      });
    });
  });

  // アコーディオン機能の実装
  const accordionHeaders = document.querySelectorAll('.setting-group-header');
  
  accordionHeaders.forEach(header => {
    header.addEventListener('click', () => {
      // ヘッダーのアクティブ状態を切り替え
      header.classList.toggle('active');
      
      // コンテンツ部分の表示/非表示を切り替え
      const content = header.nextElementSibling;
      content.classList.toggle('active');
      
      // アニメーションのためのスタイル調整
      if (content.classList.contains('active')) {
        content.style.maxHeight = content.scrollHeight + 'px';
      } else {
        content.style.maxHeight = '0px';
      }
    });
  });

  // チェックボックスの変更を監視（改善版）
  document.querySelectorAll('input[type="checkbox"]:not(#darkMode)').forEach(checkbox => {
    // ラベルクリックでチェックボックス切り替え
    const settingItem = checkbox.closest('.setting-item');
    const titleElement = settingItem?.querySelector('.setting-item-title');
    
    if (titleElement) {
      titleElement.addEventListener('click', () => {
        checkbox.click();
      });
      titleElement.style.cursor = 'pointer';
    }

    checkbox.addEventListener('change', function(e) {
      // 視覚的フィードバック
      const settingItem = e.target.closest('.setting-item');
      if (settingItem) {
        settingItem.classList.add('highlighted');
        setTimeout(() => settingItem.classList.remove('highlighted'), 300);
      }

      const setting = {};
      if (e.target.id === 'movieplus') {
        setting['includeMoviePlus'] = e.target.checked;
      } else if (e.target.id === 'autoHideNPlus') {
        setting['autoHideNPlus'] = e.target.checked;
        chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'autoHideNPlusChanged',
            autoHideNPlus: e.target.checked
          });
        });
      } else if (e.target.id === 'enableVideoEndSound') {
        setting['enableVideoEndSound'] = e.target.checked;
        chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'videoEndSoundChanged',
            enableVideoEndSound: e.target.checked
          });
        });
      } else if (e.target.id === 'showDynamicIsland') {
        setting['showDynamicIsland'] = e.target.checked;
        chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'dynamicIslandSettingChanged',
            showDynamicIsland: e.target.checked
          });
        });
      } else {
        setting[e.target.id] = e.target.checked;
      }
      saveSettingAndNotify(setting);
    });
  });

  // Discord Webhook URLの変更を監視（改善版）
  const discordWebhookInput = document.getElementById('discordWebhook');
  let webhookTimeout;

  discordWebhookInput.addEventListener('input', function(e) {
    // デバウンス処理で保存頻度を制限
    clearTimeout(webhookTimeout);
    webhookTimeout = setTimeout(() => {
      saveSettingAndNotify({ discordWebhook: e.target.value });
      
      // 視覚的フィードバック
      const settingItem = e.target.closest('.setting-item');
      if (settingItem) {
        settingItem.classList.add('highlighted');
        setTimeout(() => settingItem.classList.remove('highlighted'), 300);
      }
    }, 500); // 500ms後に保存
  });

  // URLの妥当性チェック
  discordWebhookInput.addEventListener('blur', function(e) {
    const url = e.target.value.trim();
    if (url && !url.startsWith('https://discord.com/api/webhooks/')) {
      e.target.style.borderColor = 'var(--color-error)';
      setTimeout(() => {
        e.target.style.borderColor = 'var(--color-separator)';
      }, 2000);
    }
  });

  document.getElementById('darkMode').addEventListener('change', function(e) {
    const modal = document.getElementById('darkModeModal');
    if (e.target.checked) {
      modal.style.display = "block";
    } else {
      // チェックを外した場合は設定を保存して通知
      setDarkModeSetting(false);
    }
  });

  document.getElementById('darkModeConfirm').addEventListener('click', function() {
    setDarkModeSetting(true);
    closeModal();
  });

  document.getElementById('darkModeCancel').addEventListener('click', function() {
    document.getElementById('darkMode').checked = false;
    closeModal();
  });

  // 履歴クリアボタンのイベントリスナーを追加
  document.getElementById('clearHistory').addEventListener('click', function() {
    if (confirm('本当にすべての履歴を削除しますか？')) {
      chrome.storage.sync.remove('courseHistory', function() {
        if (chrome.runtime.lastError) {
          console.error('Error clearing history:', chrome.runtime.lastError);
        } else {
          loadCourseHistory(); // 履歴を再読み込み
        }
      });
    }
  });

  // 古い履歴削除ボタンのイベントリスナー
  document.getElementById('cleanupHistory').addEventListener('click', function() {
    if (confirm('30日以上古い履歴を削除しますか？')) {
      chrome.storage.sync.get('courseHistory', function(result) {
        if (chrome.runtime.lastError) {
          console.error('Error getting history for cleanup:', chrome.runtime.lastError);
          return;
        }
        
        let courseHistory = result.courseHistory || {};
        const entries = Object.entries(courseHistory);
        const originalCount = entries.length;
        
        // 30日以上古いデータを削除
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
        
        const filteredEntries = entries.filter(([key, value]) => {
          if (!value.timestamp) return false;
          const entryDate = new Date(value.timestamp);
          return entryDate > thirtyDaysAgo;
        });
        
        if (filteredEntries.length < originalCount) {
          const cleanedHistory = Object.fromEntries(filteredEntries);
          chrome.storage.sync.set({ courseHistory: cleanedHistory }, function() {
            if (chrome.runtime.lastError) {
              console.error('Error saving cleaned history:', chrome.runtime.lastError);
            } else {
              alert(`${originalCount - filteredEntries.length}件の古い履歴を削除しました。`);
              loadCourseHistory(); // 履歴を再読み込み
            }
          });
        } else {
          alert('削除対象の古い履歴はありませんでした。');
        }
      });
    }
  });

  // 緊急クリーンアップボタンのイベントリスナー
  document.getElementById('emergencyCleanup').addEventListener('click', function() {
    if (confirm('緊急クリーンアップを実行しますか？\n最新の5件のみ保持し、データを最小限に圧縮します。')) {
      chrome.storage.sync.get('courseHistory', function(result) {
        if (chrome.runtime.lastError) {
          console.error('Error getting history for emergency cleanup:', chrome.runtime.lastError);
          return;
        }
        
        let courseHistory = result.courseHistory || {};
        const entries = Object.entries(courseHistory);
        const originalCount = entries.length;
        
        // 最新の5件のみ保持
        entries.sort((a, b) => new Date(a[1].timestamp) - new Date(b[1].timestamp));
        const recentEntries = entries.slice(-5);
        
        // データをさらに簡素化
        const minimalHistory = Object.fromEntries(
          recentEntries.map(([key, value]) => [
            key,
            {
              videoTime: value.videoTime,
              videoCount: value.videoCount,
              testCount: value.testCount,
              questionCount: value.questionCount,
              timestamp: value.timestamp?.split('T')[0] || value.timestamp // 日付のみ保存
            }
          ])
        );
        
        chrome.storage.sync.set({ courseHistory: minimalHistory }, function() {
          if (chrome.runtime.lastError) {
            console.error('Emergency cleanup failed:', chrome.runtime.lastError);
            alert('緊急クリーンアップに失敗しました。');
          } else {
            const newCount = Object.keys(minimalHistory).length;
            const newSize = JSON.stringify(minimalHistory).length;
            alert(`緊急クリーンアップ完了\n${originalCount} → ${newCount} 件\nサイズ: ${newSize} バイト`);
            loadCourseHistory(); // 履歴を再読み込み
            updateStorageUsage(); // ストレージ使用量を更新
          }
        });
      });
    }
  });

  // ストレージ使用量を表示する機能
  function updateStorageUsage() {
    chrome.storage.sync.getBytesInUse(null, function(bytesInUse) {
      if (chrome.runtime.lastError) {
        console.error('Error checking storage usage:', chrome.runtime.lastError);
        return;
      }
      
      const totalQuota = chrome.storage.sync.QUOTA_BYTES || 102400; // 100KB
      const usagePercent = (bytesInUse / totalQuota) * 100;
      const storageUsageElement = document.getElementById('storageUsage');
      
      if (storageUsageElement) {
        storageUsageElement.textContent = `ストレージ使用量: ${bytesInUse}/${totalQuota} bytes (${usagePercent.toFixed(1)}%)`;
        
        // 使用量に応じて色を変更
        if (usagePercent > 80) {
          storageUsageElement.style.color = '#dc3545'; // 赤
        } else if (usagePercent > 60) {
          storageUsageElement.style.color = '#fd7e14'; // オレンジ
        } else {
          storageUsageElement.style.color = '#666'; // グレー
        }
      }
    });
  }

  // 初期表示時にストレージ使用量を更新
  updateStorageUsage();

  function closeModal() {
    document.getElementById('darkModeModal').style.display = "none";
  }

  function setDarkModeSetting(enabled) {
    saveSettingAndNotify({ darkMode: enabled });
    chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
      chrome.tabs.sendMessage(tabs[0].id, {
        type: 'darkModeChanged',
        darkMode: enabled
      });
    });
  }

  function loadSettings() {
    chrome.storage.sync.get(['includeMoviePlus', 'autoHideNPlus', 'darkMode', 'showVideoTime', 
                            'showVideoCount', 'showTestCount', 'showQuestionCount', 
                            'enableVideoEndSound', 'discordWebhook', 'showRemainingTime',
                            'visualizeProgress', 'videoEndSound', 'videoEndVolume',
                            'breakStartSound', 'breakStartVolume', 'breakEndSound', 'breakEndVolume',
                            'chatgptTemporaryChat', 'showDynamicIsland', 'enableCharlimitValidator'], function(result) {
      if (chrome.runtime.lastError) {
        console.error('Error loading settings:', chrome.runtime.lastError);
        return;
      }
      
      document.getElementById('movieplus').checked = result.includeMoviePlus || false;
      document.getElementById('autoHideNPlus').checked = result.autoHideNPlus || false;
      document.getElementById('darkMode').checked = result.darkMode || false;
      document.getElementById('showVideoTime').checked = result.showVideoTime || false;
      document.getElementById('showVideoCount').checked = result.showVideoCount || false;
      document.getElementById('showTestCount').checked = result.showTestCount || false;
      document.getElementById('showQuestionCount').checked = result.showQuestionCount || false;
      document.getElementById('enableVideoEndSound').checked = result.enableVideoEndSound || false;
      document.getElementById('discordWebhook').value = result.discordWebhook || '';
      document.getElementById('showRemainingTime').checked = result.showRemainingTime || false;
      document.getElementById('visualizeProgress').checked = result.visualizeProgress || false;
      document.getElementById('chatgptTemporaryChat').checked = result.chatgptTemporaryChat || false;
      document.getElementById('showDynamicIsland').checked = result.showDynamicIsland !== false; // デフォルトで有効
      // 文字数・文字種チェック（デフォルト: 有効）
      const charlimitEnabled = result.enableCharlimitValidator !== false;
      const charlimitEl = document.getElementById('enableCharlimitValidator');
      if (charlimitEl) charlimitEl.checked = charlimitEnabled;
      
      // 音声設定の読み込み
      document.getElementById('videoEndSound').value = result.videoEndSound || 'notification.mp3';
      document.getElementById('videoEndVolume').value = result.videoEndVolume || 100;
      document.querySelector('#videoEndVolume + .volume-value').textContent = (result.videoEndVolume || 100) + '%';
      
      document.getElementById('breakStartSound').value = result.breakStartSound || 'notify_bright.mp3';
      document.getElementById('breakStartVolume').value = result.breakStartVolume || 100;
      document.querySelector('#breakStartVolume + .volume-value').textContent = (result.breakStartVolume || 100) + '%';
      
      document.getElementById('breakEndSound').value = result.breakEndSound || 'timer_end.mp3';
      document.getElementById('breakEndVolume').value = result.breakEndVolume || 100;
      document.querySelector('#breakEndVolume + .volume-value').textContent = (result.breakEndVolume || 100) + '%';
      
      // 初回のみコンテンツスクリプトに全設定を送信（設定が初期化されていない場合のみ）
      chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
        if (tabs[0]) {
          // まず設定初期化状態を確認
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'checkSettingsInitialized'
          }).then(response => {
            // 設定が未初期化の場合のみ全設定を送信
            if (!response || !response.initialized) {
              chrome.tabs.sendMessage(tabs[0].id, {
                type: 'allSettingsLoaded',
                settings: {
                  includeMoviePlus: result.includeMoviePlus || false,
                  autoHideNPlus: result.autoHideNPlus || false,
                  darkMode: result.darkMode || false,
                  enableVideoEndSound: result.enableVideoEndSound || false,
                  discordWebhook: result.discordWebhook || '',
                  visualizeProgress: result.visualizeProgress || false,
                  enableCharlimitValidator: charlimitEnabled,
                  videoEndSound: result.videoEndSound || 'notification.mp3',
                  videoEndVolume: result.videoEndVolume || 100,
                  breakStartSound: result.breakStartSound || 'notify_bright.mp3',
                  breakStartVolume: result.breakStartVolume || 100,
                  breakEndSound: result.breakEndSound || 'timer_end.mp3',
                  breakEndVolume: result.breakEndVolume || 100,
                  showDynamicIsland: result.showDynamicIsland !== false
                }
              }).catch(error => {
                // content script未ロード時は静かにスキップ
              });
            }
          }).catch(error => {
            // content scriptが読み込まれていない場合は静かにスキップ
          });
        }
      });
    });
  }

  function saveSettingAndNotify(setting) {
    chrome.storage.sync.set(setting, function() {
      if (chrome.runtime.lastError) {
        console.error('Error saving settings:', chrome.runtime.lastError);
        // エラー時の視覚的フィードバック
        showSettingsToast('❌ 設定の保存に失敗しました', 'error');
        return;
      }
      // デバッグログ削除: 設定保存成功
      
      // 成功時の控えめなフィードバック
      showSettingsToast('✅ 設定を保存しました', 'success');
    });

    chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
      if (tabs[0]) {
        chrome.tabs.sendMessage(tabs[0].id, {
          type: 'settingChanged',
          ...setting
        });
      }
    });
  }

  // 設定用トースト通知
  function showSettingsToast(message, type = 'info') {
    // 既存のトーストを削除
    const existingToast = document.querySelector('.settings-toast');
    if (existingToast) {
      existingToast.remove();
    }

    const toast = document.createElement('div');
    toast.className = `settings-toast ${type}`;
    toast.textContent = message;
    
    // スタイルを直接適用
    Object.assign(toast.style, {
      position: 'fixed',
      top: '10px',
      right: '10px',
      padding: '8px 12px',
      borderRadius: '6px',
      fontSize: '13px',
      fontWeight: '500',
      zIndex: '1000',
      opacity: '0',
      transform: 'translateY(-10px)',
      transition: 'all 0.3s ease',
      pointerEvents: 'none',
      maxWidth: '200px'
    });

    // タイプ別の色設定
    if (type === 'success') {
      toast.style.backgroundColor = 'rgba(52, 199, 89, 0.9)';
      toast.style.color = 'white';
    } else if (type === 'error') {
      toast.style.backgroundColor = 'rgba(255, 59, 48, 0.9)';
      toast.style.color = 'white';
    } else {
      toast.style.backgroundColor = 'rgba(0, 122, 255, 0.9)';
      toast.style.color = 'white';
    }

    document.body.appendChild(toast);

    // アニメーション表示
    setTimeout(() => {
      toast.style.opacity = '1';
      toast.style.transform = 'translateY(0)';
    }, 10);

    // 自動削除
    setTimeout(() => {
      toast.style.opacity = '0';
      toast.style.transform = 'translateY(-10px)';
      setTimeout(() => toast.remove(), 300);
    }, 2000);
  }

  function loadCourseHistory() {
    const historyContainer = document.getElementById('courseHistoryContainer');
    historyContainer.innerHTML = '<div class="loading-message">履歴を読み込み中...</div>';
    
    // ローカルとサーバーの両方からデータを読み込む
    Promise.all([
      loadLocalHistory(),
      loadServerHistory()
    ]).then(([localHistory, serverHistory]) => {
      const combinedHistory = mergeHistories(localHistory, serverHistory);
      displayHistory(combinedHistory);
    }).catch(error => {
      console.error('履歴読み込みエラー:', error);
      historyContainer.innerHTML = '<div class="no-history">履歴の読み込みに失敗しました</div>';
    });
  }
  
  function loadLocalHistory() {
    return new Promise((resolve) => {
      chrome.storage.sync.get('courseHistory', function(result) {
        if (chrome.runtime.lastError) {
          console.error('Error loading local history:', chrome.runtime.lastError);
          resolve({});
          return;
        }
        resolve(result.courseHistory || {});
      });
    });
  }
  
  async function loadServerHistory() {
    try {
      const result = await chrome.storage.local.get('jwt');
      if (typeof result.jwt !== 'string' || result.jwt.length === 0) {
        return {};
      }
      
      const response = await fetch("https://zsp-api.yoima.com/loadHistory.php", {
        headers: { 
          "Authorization": "Bearer " + result.jwt,
          "X-Auth-Token": result.jwt
        }
      });
      
      if (response.ok) {
        const serverData = await response.json();
        const convertedHistory = {};
        
        // サーバーデータをローカル形式に変換
        serverData.forEach(item => {
          const courseInfo = `${item.course_id}/chapters/${item.chapter_id}`;
          convertedHistory[courseInfo] = {
            url: item.url,
            title: item.title,
            videoTime: item.video_time,
            videoCount: item.video_count,
            testCount: item.test_count,
            questionCount: item.question_count,
            timestamp: item.updated_at,
            isFromServer: true
          };
        });
        
        return convertedHistory;
      } else if (response.status === 401) {
        // 認証エラーの場合、保存されたトークンを削除してログイン画面を表示
        console.error('認証エラー: トークンが無効です');
        await chrome.storage.local.remove(['jwt', 'username']);
        showLoginForm();
        showAuthMessage('認証の有効期限が切れました。再度ログインしてください。', 'error');
        return {};
      } else {
        const errorData = await response.json().catch(() => null);
        console.error('サーバー履歴読み込みエラー:', errorData?.error || `HTTP ${response.status}`);
        return {};
      }
    } catch (error) {
      console.error('サーバー履歴読み込みエラー:', error);
      return {};
    }
  }
  
  function mergeHistories(localHistory, serverHistory) {
    const merged = { ...localHistory };
    
    // サーバーデータで上書き（サーバーデータが最新とする）
    Object.keys(serverHistory).forEach(courseInfo => {
      const serverItem = serverHistory[courseInfo];
      const localItem = merged[courseInfo];
      
      if (!localItem || new Date(serverItem.timestamp) > new Date(localItem.timestamp)) {
        merged[courseInfo] = serverItem;
      }
    });
    
    return merged;
  }
  
  function displayHistory(courseHistory) {
    const historyContainer = document.getElementById('courseHistoryContainer');
    
    if (Object.keys(courseHistory).length === 0) {
      historyContainer.innerHTML = '<div class="no-history">履歴がありません</div>';
      return;
    }
    
    historyContainer.innerHTML = '';
    
    // 新しい順に並べ替え
    const sortedHistory = Object.entries(courseHistory).sort((a, b) => {
      return new Date(b[1].timestamp) - new Date(a[1].timestamp);
    });
    
    sortedHistory.forEach(([courseInfo, data]) => {
      const historyItem = document.createElement('div');
      historyItem.className = 'history-item';
      
      const date = new Date(data.timestamp);
      const formattedDate = `${date.getFullYear()}/${(date.getMonth()+1).toString().padStart(2, '0')}/${date.getDate().toString().padStart(2, '0')} ${date.getHours().toString().padStart(2, '0')}:${date.getMinutes().toString().padStart(2, '0')}`;
      
      const sourceIcon = data.isFromServer ? '☁️' : '💾';
      
      historyItem.innerHTML = `
        <div class="history-title">
          <a href="${data.url}" target="_blank">${data.title}</a>
          <span class="history-source">${sourceIcon}</span>
          <span class="history-date">${formattedDate}</span>
        </div>
        <div class="history-details">
          <div class="history-stat">動画時間: ${data.videoTime}</div>
          <div class="history-stat">動画数: ${data.videoCount}</div>
          <div class="history-stat">テスト数: ${data.testCount}</div>
          <div class="history-stat">問題数: ${data.questionCount}</div>
        </div>
      `;
      
      historyContainer.appendChild(historyItem);
    });
  }
  
  // アカウント機能の実装
  async function initializeAccount() {
    // まずイベントリスナーを設定
    setupAccountEventListeners();
    
    try {
      const result = await chrome.storage.local.get(['jwt', 'username']);
      if (result.jwt && result.username) {
        showLoggedInView(result.username);
      } else {
        showLoginForm();
      }
    } catch (error) {
      console.error('アカウント初期化エラー:', error);
      showLoginForm();
    }
  }
  
  function setupAccountEventListeners() {
    // ログインボタンのイベントリスナーを設定
    const loginBtn = document.getElementById('loginBtn');
    const registerBtn = document.getElementById('registerBtn');
    const syncNowBtn = document.getElementById('syncNowBtn');
    
    if (loginBtn) {
      // 既存のイベントリスナーを削除してから新しいものを追加
      loginBtn.removeEventListener('click', handleLogin);
      loginBtn.addEventListener('click', handleLogin);
    }
    
    if (registerBtn) {
      registerBtn.removeEventListener('click', handleRegister);
      registerBtn.addEventListener('click', handleRegister);
    }
    
    if (syncNowBtn) {
      syncNowBtn.removeEventListener('click', handleSyncNow);
      syncNowBtn.addEventListener('click', handleSyncNow);
    }
  }
  
  function showLoginForm() {
    document.getElementById('loginStatus').innerHTML = '<div class="status-loading">ログインしてオンライン保存を有効にする</div>';
    document.getElementById('loginForm').style.display = 'block';
    document.getElementById('loggedInView').style.display = 'none';
  }
  
  function showLoggedInView(username) {
    document.getElementById('loginStatus').innerHTML = '';
    document.getElementById('loginForm').style.display = 'none';
    document.getElementById('loggedInView').style.display = 'block';
    document.getElementById('currentUsername').textContent = username;
  }
  
  async function handleLogin() {
    const username = document.getElementById('username').value.trim();
    const password = document.getElementById('password').value.trim();
    const messageDiv = document.getElementById('authMessage');
    const loginBtn = document.getElementById('loginBtn');
    
    if (!username || !password) {
      showAuthMessage('ユーザー名とパスワードを入力してください', 'error');
      return;
    }
    
    loginBtn.disabled = true;
    loginBtn.textContent = 'ログイン中...';
    
    try {
      const response = await fetch("https://zsp-api.yoima.com/login.php", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, password })
      });
      
      if (response.ok) {
        const { token } = await response.json();
        await chrome.storage.local.set({ jwt: token, username: username });
        showAuthMessage('ログインしました', 'success');
        
        // ログイン成功時にタイマーUIを更新
        try {
          updateTimerUIBasedOnLogin(true);
        } catch (timerError) {
          console.error('タイマーUI更新エラー:', timerError);
        }
        
        setTimeout(() => {
          showLoggedInView(username);
          // ログイン後に同期を実行
          try {
            syncToServer();
          } catch (syncError) {
            console.error('同期エラー:', syncError);
          }
        }, 1000);
      } else {
        const error = await response.json();
        showAuthMessage(error.error || 'ログインに失敗しました', 'error');
      }
    } catch (error) {
      console.error('ログインエラー:', error);
      if (error.message.includes('Could not establish connection')) {
        showAuthMessage('拡張機能の接続エラーが発生しました。拡張機能を再読み込みしてください。', 'error');
      } else {
        showAuthMessage('ネットワークエラーが発生しました', 'error');
      }
    }
    
    loginBtn.disabled = false;
    loginBtn.textContent = 'ログイン';
  }
  
  async function handleRegister() {
    const username = document.getElementById('username').value.trim();
    const password = document.getElementById('password').value.trim();
    const registerBtn = document.getElementById('registerBtn');
    
    if (!username || !password) {
      showAuthMessage('ユーザー名とパスワードを入力してください', 'error');
      return;
    }
    
    if (password.length < 6) {
      showAuthMessage('パスワードは6文字以上で入力してください', 'error');
      return;
    }
    
    registerBtn.disabled = true;
    registerBtn.textContent = '登録中...';
    
    try {
      const response = await fetch("https://zsp-api.yoima.com/register.php", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, password })
      });
      
      if (response.ok) {
        showAuthMessage('アカウントが作成されました。ログインしてください。', 'success');
        document.getElementById('password').value = '';
      } else {
        const error = await response.json();
        let errorMessage = error.error || '登録に失敗しました';
        if (error.error === 'Username already exists') {
          errorMessage = 'このユーザー名は既に使用されています。別のユーザー名をお試しください。';
        }
        showAuthMessage(errorMessage, 'error');
      }
    } catch (error) {
      console.error('登録エラー:', error);
      showAuthMessage('ネットワークエラーが発生しました', 'error');
    }
    
    registerBtn.disabled = false;
    registerBtn.textContent = '新規登録';
  }
  
  async function handleSyncNow() {
    const syncBtn = document.getElementById('syncNowBtn');
    syncBtn.disabled = true;
    syncBtn.textContent = '同期中...';
    
    try {
      await syncToServer();
      showSyncMessage('同期が完了しました', 'success');
    } catch (error) {
      console.error('同期エラー:', error);
      const errorMessage = error.message || '同期に失敗しました';
      showSyncMessage(errorMessage, 'error');
    }
    
    syncBtn.disabled = false;
    syncBtn.textContent = '今すぐ同期';
  }
  
  async function syncToServer() {
    try {
      const storageData = await chrome.storage.local.get('jwt');
      const syncData = await chrome.storage.sync.get('courseHistory');
      
      if (typeof storageData.jwt !== 'string' || storageData.jwt.length === 0 || !syncData.courseHistory) {
        return;
      }
      
      // ローカルデータをAPI形式に変換
      const courseHistory = syncData.courseHistory;
      const arrayOfLessonData = Object.entries(courseHistory).map(([courseInfo, data]) => {
        const match = courseInfo.match(/(\d+)\/chapters\/(\d+)/);
        if (!match) return null;
        
        return {
          course_id: match[1],
          chapter_id: match[2],
          url: data.url,
          title: data.title,
          video_time: data.videoTime,
          video_time_s: convertTimeToSeconds(data.videoTime),
          video_count: data.videoCount,
          test_count: data.testCount,
          question_count: data.questionCount,
          updated_at: data.timestamp
        };
      }).filter(item => item !== null);
      
      if (arrayOfLessonData.length === 0) {
        return;
      }
      
      const response = await fetch("https://zsp-api.yoima.com/saveHistory.php", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Authorization": "Bearer " + storageData.jwt,
          "X-Auth-Token": storageData.jwt
        },
        body: JSON.stringify(arrayOfLessonData)
      });
      
      if (response.status === 401) {
        // 認証エラーの場合、保存されたトークンを削除してログイン画面を表示
        await chrome.storage.local.remove(['jwt', 'username']);
        showLoginForm();
        showAuthMessage('認証の有効期限が切れました。再度ログインしてください。', 'error');
        const errorData = await response.json().catch(() => null);
        throw new Error(errorData?.error || '認証エラー');
      } else if (!response.ok) {
        const errorData = await response.json().catch(() => null);
        const message = errorData?.error || `サーバーエラー: ${response.status}`;
        throw new Error(message);
      }
      
      // 成功した場合も履歴を再読み込みして表示を更新
      loadCourseHistory();
      
    } catch (error) {
      // console.errorはここで出力せず、呼び出し元で出力する
      throw error;
    }
  }
  
  function convertTimeToSeconds(timeString) {
    const match = timeString.match(/(\d+)分(\d+)秒/);
    if (match) {
      return parseInt(match[1]) * 60 + parseInt(match[2]);
    }
    return 0;
  }
  
  function showAuthMessage(message, type) {
    const messageDiv = document.getElementById('authMessage');
    messageDiv.textContent = message;
    messageDiv.className = `auth-message ${type}`;
    setTimeout(() => {
      messageDiv.textContent = '';
      messageDiv.className = 'auth-message';
    }, 3000);
  }
  
  function showSyncMessage(message, type) {
    const messageDiv = document.getElementById('syncMessage');
    messageDiv.textContent = message;
    messageDiv.className = `sync-message ${type}`;
    setTimeout(() => {
      messageDiv.textContent = '';
      messageDiv.className = 'sync-message';
    }, 3000);
  }

  // ---------------- ログイン状態チェック ----------------
  
  // ログイン状態をチェックしてタイマーUIを制御する関数
  async function checkLoginStatusForTimer() {
    try {
      const result = await chrome.storage.local.get('jwt');
      const isLoggedIn = result.jwt && result.jwt.length > 0;
      
      updateTimerUIBasedOnLogin(isLoggedIn);
    } catch (error) {
      console.error('ログイン状態の確認でエラーが発生しました:', error);
      updateTimerUIBasedOnLogin(false);
    }
  }
  
  // ログイン状態に基づいてタイマーUIを更新
  function updateTimerUIBasedOnLogin(isLoggedIn) {
    const focusControls = document.querySelector('.focus-controls');
    const loginRequiredOverlay = document.getElementById('loginRequiredOverlay');
    
    if (isLoggedIn) {
      // ログイン済み：オーバーレイを非表示、タイマーを有効化
      focusControls.classList.remove('login-blocked');
      loginRequiredOverlay.style.display = 'none';
    } else {
      // 未ログイン：オーバーレイを表示、タイマーを無効化
      focusControls.classList.add('login-blocked');
      loginRequiredOverlay.style.display = 'flex';
    }
  }
  
  // ログインページへ移動するボタンのイベントリスナー
  document.getElementById('goToLoginTab').addEventListener('click', function() {
    // 同期タブに切り替え
    const accountTab = document.querySelector('[data-tab="account"]');
    accountTab.click();
  });

  // ---------------- フォーカスタイマー UI ----------------
  // タイマーUI要素を取得
  const startFocusBtn = document.getElementById('startFocus');
  const stopFocusBtn  = document.getElementById('stopFocus');
  const elapsedLabel  = document.getElementById('focusElapsed');
  
  let todayTotalMinutes = 0; // 今日の累計時間を保持
  let sessionStartTime = 0;  // 現在のセッション開始時刻

  function formatDuration(ms) {
    const totalSec = Math.floor(ms / 1000);
    const h = String(Math.floor(totalSec / 3600)).padStart(2, '0');
    const m = String(Math.floor((totalSec % 3600) / 60)).padStart(2, '0');
    const s = String(totalSec % 60).padStart(2, '0');
    return `${h}:${m}:${s}`;
  }

  function formatMinutesToTime(minutes) {
    const h = String(Math.floor(minutes / 60)).padStart(2, '0');
    const m = String(minutes % 60).padStart(2, '0');
    return `${h}:${m}:00`;
  }

  function updateElapsedDisplay(state) {
    const elapsedLabel = document.getElementById('focusElapsed');
    const timerLabel = document.getElementById('timerLabel');
    const progressCircle = document.getElementById('progressCircle');
    
    // 手動モードの場合はシンプルタイマーを更新
    const simpleTimerTime = document.getElementById('simpleTimerTime');
    const simpleTimerLabel = document.getElementById('simpleTimerLabel');
    
    if (!state || !state.active) {
      elapsedLabel.textContent = '00:00';
      timerLabel.textContent = '集中時間';
      simpleTimerTime.textContent = '00:00';
      simpleTimerLabel.textContent = '集中時間';
      
      // Reset progress circle
      if (progressCircle) {
        progressCircle.style.strokeDashoffset = '339.292';
      }
    } else {
      // 現在のセッションの経過分数を計算
      const sessionElapsedMs = state.currentElapsedMs || 0;
      const sessionElapsedMinutes = Math.floor(sessionElapsedMs / (1000 * 60));
      const sessionElapsedSeconds = Math.floor((sessionElapsedMs % (1000 * 60)) / 1000);
      
      // MM:SS format for compact display
      const m = String(sessionElapsedMinutes).padStart(2, '0');
      const s = String(sessionElapsedSeconds).padStart(2, '0');
      const timeString = `${m}:${s}`;
      
      elapsedLabel.textContent = timeString;
      timerLabel.textContent = '学習中';
      
      // 手動モードの場合はシンプルタイマーも更新
      simpleTimerTime.textContent = timeString;
      simpleTimerLabel.textContent = '学習中';
      
      // Update progress circle for manual timer (showing continuous progress)
      if (progressCircle) {
        // For manual timer, show a slow continuous progress
        const totalMinutes = sessionElapsedMinutes + 1; // Add 1 to avoid division by zero
        const maxMinutes = 25; // Use 25 minutes as reference for one full circle
        const progress = Math.min(totalMinutes / maxMinutes, 1);
        const circumference = 339.292;
        const offset = circumference - (progress * circumference);
        progressCircle.style.strokeDashoffset = offset;
      }
    }
  }

  function updateFocusUI(state) {
    if (!state || !state.active) {
      startFocusBtn.disabled = false;
      stopFocusBtn.disabled  = true;
      updateElapsedDisplay(state);
    } else {
      startFocusBtn.disabled = true;
      stopFocusBtn.disabled  = false;
      updateElapsedDisplay(state);
    }
  }

  function updateAdvancedTimerUI(state) {
    const elapsedLabel = document.getElementById('focusElapsed');
    const timerLabel = document.getElementById('timerLabel');
    const progressCircle = document.getElementById('progressCircle');
    
    if (!state || !state.active) {
      startFocusBtn.disabled = false;
      stopFocusBtn.disabled  = true;
      skipToBreakBtn.style.display = 'none';
      skipBreakBtn.style.display = 'none';
      elapsedLabel.textContent = '00:00';
      timerLabel.textContent = '集中時間';
      
      // Reset progress circle
      if (progressCircle) {
        progressCircle.style.strokeDashoffset = '339.292';
      }
    } else {
      startFocusBtn.disabled = true;
      stopFocusBtn.disabled  = false;
      
      // スキップボタンの表示制御
      if (state.phase === 'focus') {
        skipToBreakBtn.style.display = 'inline-block';
        skipBreakBtn.style.display = 'none';
      } else {
        skipToBreakBtn.style.display = 'none';
        skipBreakBtn.style.display = 'inline-block';
      }
      
      // 残り時間を表示
      if (state.remainSec !== undefined) {
        const m = String(Math.floor(state.remainSec / 60)).padStart(2, '0');
        const s = String(state.remainSec % 60).padStart(2, '0');
        elapsedLabel.textContent = `${m}:${s}`;
        
        // フェーズ情報の表示
        const phaseText = state.phase === 'focus' ? '集中' : 
                         state.phase === 'short_break' ? '短休憩' : '長休憩';
        timerLabel.textContent = `${phaseText} - ${state.cycle}`;
        
        // Update progress circle for advanced timer
        if (progressCircle) {
          let totalDuration;
          if (state.phase === 'focus') {
            totalDuration = timerMode === 'pomodoro' ? 25 * 60 : 45 * 60; // 25 or 45 minutes
          } else {
            totalDuration = timerMode === 'pomodoro' ? 5 * 60 : 15 * 60; // 5 or 15 minutes
          }
          
          const elapsed = totalDuration - state.remainSec;
          const progress = elapsed / totalDuration;
          const circumference = 339.292;
          const offset = circumference - (progress * circumference);
          progressCircle.style.strokeDashoffset = offset;
          
          // Change color based on phase
          if (state.phase === 'focus') {
            progressCircle.style.stroke = 'var(--color-primary)';
          } else {
            progressCircle.style.stroke = 'var(--color-success)';
          }
        }
      }
    }
  }

  async function fetchStatus() {
    if (timerMode === 'manual') {
      chrome.runtime.sendMessage({ type: 'studyStatus' }, (state) => {
        updateFocusUI(state);
      });
    } else {
      fetchAdvancedStatus();
    }
  }

  async function fetchAdvancedStatus() {
    chrome.runtime.sendMessage({ type: 'advancedTimerStatus' }, (state) => {
      updateAdvancedTimerUI(state);
    });
  }

  startFocusBtn.addEventListener('click', async () => {
    // ログイン状態をチェック
    const result = await chrome.storage.local.get('jwt');
    const isLoggedIn = result.jwt && result.jwt.length > 0;
    
    if (!isLoggedIn) {
      // ログインしていない場合は、同期タブに切り替えて警告を表示
      const accountTab = document.querySelector('[data-tab="account"]');
      accountTab.click();
      showAuthMessage('タイマー機能を使用するにはログインが必要です', 'error');
      return;
    }
    
    if (timerMode === 'manual') {
      // 手動タイマーの場合
      chrome.runtime.sendMessage({ type: 'studyStart' }, (response) => {
        if (response && response.ok) {
          // 累計時間を保存
          todayTotalMinutes = response.todayMinutes || 0;
          sessionStartTime = Date.now();
        }
        fetchStatus();
      });
    } else {
      // ポモドーロ・45-15タイマーの場合
      chrome.runtime.sendMessage({ type: 'advancedTimerStart', mode: timerMode }, (response) => {
        if (response && response.ok) {
          sessionStartTime = Date.now();
        }
        fetchAdvancedStatus();
      });
    }
  });

  stopFocusBtn.addEventListener('click', () => {
    if (timerMode === 'manual') {
      // 手動タイマーの場合
      chrome.runtime.sendMessage({ type: 'studyStop' }, () => {
        todayTotalMinutes = 0; // 停止時にリセット
        sessionStartTime = 0;
        fetchStatus();
      });
    } else {
      // ポモドーロ・45-15タイマーの場合
      chrome.runtime.sendMessage({ type: 'advancedTimerStop' }, () => {
        sessionStartTime = 0;
        fetchAdvancedStatus();
      });
    }
  });

  // 音声設定のイベントリスナー
  setupSoundSettings();

  // 初期化 & 1 秒ごと更新
  fetchStatus();
  const pollInterval = setInterval(() => {
    if (timerMode === 'manual') {
      fetchStatus();
    } else {
      fetchAdvancedStatus();
    }
  }, 1000);

  // ブロードキャストをリッスンしてUIを即時更新
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'STATUS_BROADCAST') {
      const { kind, payload } = message;
      if (kind === 'timer') {
        if (payload.mode === 'manual') {
          updateFocusUI(payload);
        } else {
          updateAdvancedTimerUI(payload);
        }
      } else if (kind === 'lofi') {
        updateLofiUI(payload);
      } else if (kind === 'ambient') {
        updateAmbientUI(payload);
      }
    }
  });

  // ウィンドウが閉じられたらポーリングを停止
  window.addEventListener('unload', () => {
    clearInterval(pollInterval);
  });
});

// 音声設定のセットアップ
function setupSoundSettings() {
  // 音声選択の変更イベント
  ['videoEndSound', 'breakStartSound', 'breakEndSound'].forEach(soundId => {
    const selectElement = document.getElementById(soundId);
    if (selectElement) {
      selectElement.addEventListener('change', function(e) {
        const setting = {};
        setting[soundId] = e.target.value;
        chrome.storage.sync.set(setting);
        
        // content.jsに設定変更を通知
        chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'soundSettingsChanged',
            [soundId]: e.target.value
          });
        });
      });
    }
  });

  // 音量スライダーの変更イベント
  ['videoEndVolume', 'breakStartVolume', 'breakEndVolume'].forEach(volumeId => {
    const sliderElement = document.getElementById(volumeId);
    if (sliderElement) {
      sliderElement.addEventListener('input', function(e) {
        const value = parseInt(e.target.value);
        const valueElement = sliderElement.parentElement.querySelector('.volume-value');
        if (valueElement) {
          valueElement.textContent = value + '%';
        }
        
        const setting = {};
        setting[volumeId] = value;
        chrome.storage.sync.set(setting);
        
        // content.jsに設定変更を通知
        chrome.tabs.query({ active: true, currentWindow: true }, function(tabs) {
          chrome.tabs.sendMessage(tabs[0].id, {
            type: 'soundSettingsChanged',
            [volumeId]: value
          });
        });
      });
    }
  });

  // 音声テストボタンのイベント
  const testButtons = document.querySelectorAll('.test-sound-btn');
  testButtons.forEach(button => {
    button.addEventListener('click', async function(e) {
      e.preventDefault();
      const soundType = this.getAttribute('data-sound');
      await testSound(soundType);
    });
  });
}

// 音声テスト機能
async function testSound(soundType) {
  try {
    let soundFile, volume;
    
    switch(soundType) {
      case 'videoEndSound':
        soundFile = document.getElementById('videoEndSound').value;
        volume = parseInt(document.getElementById('videoEndVolume').value) / 100;
        break;
      case 'breakStartSound':
        soundFile = document.getElementById('breakStartSound').value;
        volume = parseInt(document.getElementById('breakStartVolume').value) / 100;
        break;
      case 'breakEndSound':
        soundFile = document.getElementById('breakEndSound').value;
        volume = parseInt(document.getElementById('breakEndVolume').value) / 100;
        break;
      default:
        console.error('Unknown sound type:', soundType);
        return;
    }

    // オフスクリーンドキュメントで音声を再生
    await playTestSoundOffscreen(soundFile, volume);
    
  } catch (error) {
    console.error('音声テストエラー:', error);
  }
}

// オフスクリーンで音声テストを再生
async function playTestSoundOffscreen(soundFile, volume) {
  try {
    // オフスクリーンドキュメントが存在するかチェック
    const existingContexts = await chrome.runtime.getContexts({
      contextTypes: ['OFFSCREEN_DOCUMENT']
    });

    if (existingContexts.length === 0) {
      // オフスクリーンドキュメントを作成
      await chrome.offscreen.createDocument({
        url: 'offscreen.html',
        reasons: ['AUDIO_PLAYBACK'],
        justification: '音声設定のテスト再生のため'
      });
    }

    // オフスクリーンドキュメントに音声再生を依頼
    await chrome.runtime.sendMessage({ 
      type: 'playTestSound',
      soundFile: soundFile,
      volume: volume
    });
    
  } catch (error) {
    console.error('オフスクリーン音声テストエラー:', error);
  }
}

  // 音楽機能の初期化
  initializeMusicFeatures();

  // Spotify機能の初期化（音楽モーダル内）
  initializeSpotifyFeatures();

  // Spotify設定の初期化
  initializeSpotifySettings();

function initializeMusicFeatures() {
  const musicPanelToggle = document.getElementById('musicPanelToggle');
  const musicModal = document.getElementById('musicModal');
  const musicModalClose = document.getElementById('musicModalClose');
  const musicModalOverlay = document.querySelector('.music-modal-overlay');
  const musicTabs = document.querySelectorAll('.music-tab');
  const musicTabContents = document.querySelectorAll('.music-tab-content');
  
  // 楽曲リスト
  const lofiTracks = [
    { file: '01_First_Snow.mp3', title: 'First Snow' },
    { file: '02_Laundry_On_The_Wire.mp3', title: 'Laundry On The Wire' },
    { file: '03_Everything_You_Ever_Dreamed.mp3', title: 'Everything You Ever Dreamed' },
    { file: '04_Keeping_Cool.mp3', title: 'Keeping Cool' },
    { file: '05_Snow_Drift.mp3', title: 'Snow Drift' },
    { file: '06_Windows_Down.mp3', title: 'Windows Down' },
    { file: '07_Hour_Delay.mp3', title: 'Hour Delay' }
  ];
  
  // UI要素
  const lofiPlayPause = document.getElementById('lofiPlayPause');
  const lofiStop = document.getElementById('lofiStop');
  const lofiPrevious = document.getElementById('lofiPrevious');
  const lofiNext = document.getElementById('lofiNext');
  const lofiVolume = document.getElementById('lofiVolume');
  const currentTrackTitle = document.getElementById('currentTrackTitle');
  const trackNumber = document.getElementById('trackNumber');
  const ambientSliders = document.querySelectorAll('.ambient-slider');
  const ambientStopAll = document.getElementById('ambientStopAll');

  let isLofiPlaying = false;
  let currentTrackIndex = -1; // -1は未選択状態
  let activeAmbientSounds = new Map();
  let ambientTimeouts = new Map(); // デバウンス用のタイムアウトを管理

  // 初期値の設定
  const volumeValueElement = lofiVolume.parentElement.querySelector('.volume-value');
  if (volumeValueElement) {
    volumeValueElement.textContent = lofiVolume.value;
  }
  
  // 楽曲情報を更新する関数
  function updateTrackInfo() {
    if (currentTrackIndex >= 0 && currentTrackIndex < lofiTracks.length) {
      currentTrackTitle.textContent = lofiTracks[currentTrackIndex].title;
      trackNumber.textContent = `${currentTrackIndex + 1} / ${lofiTracks.length}`;
      
      lofiPlayPause.disabled = false;
      lofiStop.disabled = false;
      lofiPrevious.disabled = false;
      lofiNext.disabled = false;
    } else {
      currentTrackTitle.textContent = '楽曲を選択...';
      trackNumber.textContent = '-- / --';
      
      lofiPlayPause.disabled = true;
      lofiStop.disabled = true;
      lofiPrevious.disabled = true;
      lofiNext.disabled = true;
    }
  }
  
  // 初期化時に最初の楽曲を選択
  if (lofiTracks.length > 0) {
    currentTrackIndex = 0;
    updateTrackInfo();
  }

  // モーダルの表示
  musicPanelToggle.addEventListener('click', () => {
    musicModal.style.display = 'flex';
    // アニメーション用のクラス追加
    setTimeout(() => {
      musicModal.classList.add('show');
    }, 10);
    // 復元: 前回選択タブ
    chrome.storage.sync.get('musicTab', (res) => {
      const tab = res.musicTab || 'lofi';
      setActiveMusicTab(tab);
    });
  });

  // モーダルを閉じる関数
  function closeModal() {
    musicModal.classList.remove('show');
    setTimeout(() => {
      musicModal.style.display = 'none';
    }, 300);
  }

  // 閉じるボタンクリック
  musicModalClose.addEventListener('click', closeModal);

  // オーバーレイクリック
  musicModalOverlay.addEventListener('click', closeModal);

  // ESCキーでモーダルを閉じる
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && musicModal.style.display === 'flex') {
      closeModal();
    }
  });

  // 音楽タブ切替
  function setActiveMusicTab(tab) {
    musicTabs.forEach(b => b.classList.toggle('active', b.dataset.tab === tab));
    musicTabContents.forEach(c => c.classList.toggle('active', c.getAttribute('data-tab-content') === tab));
    chrome.storage.sync.set({ musicTab: tab });
    // 軽い同期: Spotify側の表示更新
    if (tab === 'spotify') {
      try { initializeSpotifyFeatures._refresh && initializeSpotifyFeatures._refresh(); } catch (e) {}
    }
  }
  musicTabs.forEach(btn => btn.addEventListener('click', () => setActiveMusicTab(btn.dataset.tab)));

  // 前の楽曲に移動
  lofiPrevious.addEventListener('click', () => {
    if (currentTrackIndex > 0) {
      currentTrackIndex--;
    } else {
      // ループして最後の楽曲へ
      currentTrackIndex = lofiTracks.length - 1;
    }
    updateTrackInfo();
    playLofi();
  });

  // 次の楽曲に移動
  lofiNext.addEventListener('click', () => {
    if (currentTrackIndex < lofiTracks.length - 1) {
      currentTrackIndex++;
    } else {
      // ループして最初の楽曲へ
      currentTrackIndex = 0;
    }
    updateTrackInfo();
    playLofi();
  });

  // Lo-Fi音楽の再生/一時停止
  lofiPlayPause.addEventListener('click', () => {
    if (isLofiPlaying) {
      pauseLofi();
    } else {
      playLofi();
    }
  });

  function updateLofiUI(state) {
    isLofiPlaying = state.isPlaying;
    currentTrackIndex = state.trackIndex;
    lofiVolume.value = state.volume;
    
    const volumeValueElement = lofiVolume.parentElement.querySelector('.volume-value');
    if (volumeValueElement) {
      volumeValueElement.textContent = state.volume;
    }

    if (isLofiPlaying) {
      lofiPlayPause.innerHTML = '<span>⏸</span>';
    } else {
      lofiPlayPause.innerHTML = '<span>▶</span>';
    }
    updateTrackInfo();
  }

  function updateAmbientUI(state) {
    ambientSliders.forEach(slider => {
      const fileName = slider.dataset.file;
      const volume = state.volumes[fileName] || 0;
      slider.value = volume;
      slider.style.setProperty('--progress', `${volume}%`);
      
      if (volume > 0) {
        slider.classList.add('active');
        activeAmbientSounds.set(fileName, volume);
      } else {
        slider.classList.remove('active');
        activeAmbientSounds.delete(fileName);
      }
    });
  }

  // Lo-Fi音楽の停止
  lofiStop.addEventListener('click', () => {
    stopLofi();
  });

  // Lo-Fi音楽の音量調整
  lofiVolume.addEventListener('input', () => {
    const volume = parseInt(lofiVolume.value);
    const volumeValueElement = lofiVolume.parentElement.querySelector('.volume-value');
    if (volumeValueElement) {
      volumeValueElement.textContent = volume;
    }
    updateLofiVolume(volume);
  });

  // 環境音のスライダー
  ambientSliders.forEach(slider => {
    slider.addEventListener('input', () => {
      const volume = parseInt(slider.value);
      const fileName = slider.dataset.file;
      
      // プログレスバーの視覚的更新
      slider.style.setProperty('--progress', `${volume}%`);
      
      // 既存のタイムアウトをクリア（デバウンス処理）
      if (ambientTimeouts.has(fileName)) {
        clearTimeout(ambientTimeouts.get(fileName));
      }
      
      // 音量が0の場合は即座に停止（デバウンス不要）
      if (volume === 0) {
        slider.classList.remove('active');
        stopAmbientSound(fileName);
        return;
      }
      
      // 200ms後に実際の処理を実行（0以外の場合のみ）
      const timeoutId = setTimeout(async () => {
        slider.classList.add('active');
        // 既に再生中の場合は音量調整、そうでなければ新規再生
        if (activeAmbientSounds.has(fileName)) {
          await setAmbientVolume(fileName, volume);
        } else {
          await playAmbientSound(fileName, volume);
        }
        
        // タイムアウトをクリア
        ambientTimeouts.delete(fileName);
      }, 200);
      
      ambientTimeouts.set(fileName, timeoutId);
    });
  });

  // 環境音の全停止
  ambientStopAll.addEventListener('click', () => {
    // 全てのタイムアウトをクリア
    ambientTimeouts.forEach((timeoutId) => {
      clearTimeout(timeoutId);
    });
    ambientTimeouts.clear();
    
    ambientSliders.forEach(slider => {
      slider.value = 0;
      slider.classList.remove('active');
      slider.style.setProperty('--progress', '0%');
    });
    stopAllAmbientSounds();
  });

  async function playLofi() {
    try {
      if (currentTrackIndex < 0 || currentTrackIndex >= lofiTracks.length) {
        console.error('無効な楽曲インデックス:', currentTrackIndex);
        return;
      }
      
      await ensureOffscreenDocument();
      
      // UIを即時更新
      isLofiPlaying = true;
      updateLofiUI({
        isPlaying: true,
        trackIndex: currentTrackIndex,
        volume: parseInt(lofiVolume.value)
      });

      // バックグラウンドに再生を依頼
      await chrome.runtime.sendMessage({
        type: 'playLofi',
        fileName: lofiTracks[currentTrackIndex].file,
        volume: parseInt(lofiVolume.value),
        trackIndex: currentTrackIndex
      });
    } catch (error) {
      console.error('Lo-Fi再生エラー:', error);
      // エラー発生時はUIを元に戻す
      isLofiPlaying = false;
      updateLofiUI({
        isPlaying: false,
        trackIndex: currentTrackIndex,
        volume: parseInt(lofiVolume.value)
      });
    }
  }

  async function pauseLofi() {
    try {
      // UI を即座に更新（フォールバック）
      isLofiPlaying = false;
      updateLofiUI({
        isPlaying: false,
        trackIndex: currentTrackIndex,
        volume: parseInt(lofiVolume.value)
      });
      
      await chrome.runtime.sendMessage({ type: 'pauseLofi' });
    } catch (error) {
      console.error('Lo-Fi一時停止エラー:', error);
      // エラー時もUIを確実に更新
      isLofiPlaying = false;
      updateLofiUI({
        isPlaying: false,
        trackIndex: currentTrackIndex,
        volume: parseInt(lofiVolume.value)
      });
    }
  }

  async function stopLofi() {
    try {
      // UI を即座に更新（フォールバック）
      isLofiPlaying = false;
      updateLofiUI({
        isPlaying: false,
        trackIndex: currentTrackIndex,
        volume: parseInt(lofiVolume.value)
      });
      
      await chrome.runtime.sendMessage({ type: 'stopLofi' });
    } catch (error) {
      console.error('Lo-Fi停止エラー:', error);
      // エラー時もUIを確実に更新
      isLofiPlaying = false;
      updateLofiUI({
        isPlaying: false,
        trackIndex: currentTrackIndex,
        volume: parseInt(lofiVolume.value)
      });
    }
  }

  async function updateLofiVolume(volume) {
    try {
      await chrome.runtime.sendMessage({
        type: 'setLofiVolume',
        volume: volume
      });
    } catch (error) {
      console.error('Lo-Fi音量調整エラー:', error);
    }
  }

  async function setAmbientVolume(fileName, volume) {
    try {
      await chrome.runtime.sendMessage({
        type: 'setAmbientVolume',
        fileName: fileName,
        volume: volume
      });
      
      if (volume === 0) {
        activeAmbientSounds.delete(fileName);
      } else {
        activeAmbientSounds.set(fileName, volume);
      }
    } catch (error) {
      console.error('環境音音量調整エラー:', error);
    }
  }

  async function playAmbientSound(fileName, volume) {
    try {
      // 既に再生中の場合は重複を防ぐ
      if (activeAmbientSounds.has(fileName)) {
        await setAmbientVolume(fileName, volume);
        return;
      }
      
      await ensureOffscreenDocument();
      
      await chrome.runtime.sendMessage({
        type: 'playAmbient',
        fileName: fileName,
        volume: volume
      });
      
      activeAmbientSounds.set(fileName, volume);
    } catch (error) {
      console.error('環境音再生エラー:', error);
    }
  }

  async function stopAmbientSound(fileName) {
    try {
      await chrome.runtime.sendMessage({
        type: 'stopAmbient',
        fileName: fileName
      });
      
      activeAmbientSounds.delete(fileName);
    } catch (error) {
      console.error('環境音停止エラー:', error);
    }
  }

  async function stopAllAmbientSounds() {
    try {
      await chrome.runtime.sendMessage({
        type: 'stopAllAmbient'
      });
      
      activeAmbientSounds.clear();
    } catch (error) {
      console.error('環境音全停止エラー:', error);
    }
  }

  async function ensureOffscreenDocument() {
    const existingContexts = await chrome.runtime.getContexts({
      contextTypes: ['OFFSCREEN_DOCUMENT']
    });

    if (existingContexts.length === 0) {
      await chrome.offscreen.createDocument({
        url: 'offscreen.html',
        reasons: ['AUDIO_PLAYBACK'],
        justification: '音楽と環境音の再生のため'
      });
    }
  }
}

// Spotify Remote initialization
function initializeSpotifyFeatures() {
  const authView = document.getElementById('spotifyAuthView');
  const playerView = document.getElementById('spotifyPlayerView');
  const connectBtn = document.getElementById('spotifyConnectBtn');
  const goToSettingsBtn = document.getElementById('spotifyGoToSettings');
  const noClientIdView = document.getElementById('spotifyNoClientId');
  const readyToConnectView = document.getElementById('spotifyReadyToConnect');
  const hint = document.getElementById('spotifyAuthHint');
  const cover = document.getElementById('spotifyCover');
  const title = document.getElementById('spotifyTitle');
  const artist = document.getElementById('spotifyArtist');
  const device = document.getElementById('spotifyDevice');
  const prevBtn = document.getElementById('spotifyPrev');
  const nextBtn = document.getElementById('spotifyNext');
  const toggleBtn = document.getElementById('spotifyToggle');
  const volume = document.getElementById('spotifyVolume');

  async function refreshStatus() {
    try {
      const resp = await chrome.runtime.sendMessage({ action: 'SPOTIFY_STATUS' });
      if (!resp || !resp.success) throw new Error(resp?.error || 'status failed');
      
      if (!resp.hasClientId) {
        // クライアントIDが未設定
        authView.style.display = 'block';
        playerView.style.display = 'none';
        noClientIdView.style.display = 'block';
        readyToConnectView.style.display = 'none';
        hint.textContent = '';
        return;
      }
      
      if (!resp.authed) {
        // クライアントIDは設定済みだが未認証
        authView.style.display = 'block';
        playerView.style.display = 'none';
        noClientIdView.style.display = 'none';
        readyToConnectView.style.display = 'block';
        hint.textContent = resp.redirect ? `リダイレクトURL: ${resp.redirect}` : '';
        return;
      }
      
      // 認証済み
      authView.style.display = 'none';
      playerView.style.display = 'block';

      const p = resp.player;
      if (p) {
        const item = p.item || {};
        title.textContent = item.name || '—';
        if (item.artists && item.artists.length) {
          artist.textContent = item.artists.map(a => a.name).join(', ');
        } else {
          artist.textContent = '—';
        }
        const img = item.album?.images?.[item.album.images.length - 1] || item.album?.images?.[0];
        if (img?.url) {
          cover.src = img.url;
          cover.onerror = () => {
            cover.src = chrome.runtime.getURL('images/music.png');
          };
        } else {
          cover.src = chrome.runtime.getURL('images/music.png');
        }
        device.textContent = p.device ? `デバイス: ${p.device.name}` : '';
        toggleBtn.innerHTML = p.is_playing ? '<span>⏸</span>' : '<span>▶</span>';
      }
    } catch (e) {
      authView.style.display = 'block';
      playerView.style.display = 'none';
      noClientIdView.style.display = 'none';
      readyToConnectView.style.display = 'block';
      hint.textContent = e.message;
    }
  }

  connectBtn?.addEventListener('click', async () => {
    connectBtn.disabled = true;
    connectBtn.textContent = '接続中...';
    try {
      const r = await chrome.runtime.sendMessage({ action: 'SPOTIFY_AUTH' });
      if (!r || !r.success) throw new Error(r?.error || 'auth failed');
      await refreshStatus();
    } catch (e) {
      hint.textContent = `エラー: ${e.message}`;
    } finally {
      connectBtn.disabled = false;
      connectBtn.textContent = 'Spotify に接続';
    }
  });

  goToSettingsBtn?.addEventListener('click', () => {
    // 設定タブに切り替え
    const settingsTab = document.querySelector('[data-tab="settings"]');
    const musicModal = document.getElementById('musicModal');
    if (settingsTab && musicModal) {
      musicModal.style.display = 'none';
      settingsTab.click();
      
      // Spotify設定セクションまでスクロール
      setTimeout(() => {
        scrollToSpotifySettings();
      }, 100);
    }
  });

  prevBtn?.addEventListener('click', async () => {
    await chrome.runtime.sendMessage({ action: 'SPOTIFY_PREV' });
    setTimeout(refreshStatus, 500);
  });
  nextBtn?.addEventListener('click', async () => {
    await chrome.runtime.sendMessage({ action: 'SPOTIFY_NEXT' });
    setTimeout(refreshStatus, 500);
  });
  toggleBtn?.addEventListener('click', async () => {
    await chrome.runtime.sendMessage({ action: 'SPOTIFY_TOGGLE' });
    setTimeout(refreshStatus, 400);
  });
  volume?.addEventListener('input', async () => {
    const v = parseInt(volume.value, 10);
    // Debounce a little by delaying the call.
    clearTimeout(volume._t);
    volume._t = setTimeout(async () => {
      await chrome.runtime.sendMessage({ action: 'SPOTIFY_VOLUME', percent: v });
    }, 150);
  });

  // initial
  refreshStatus();
  // expose for tab sync
  initializeSpotifyFeatures._refresh = refreshStatus;
  // periodic refresh while modal open
  const musicModal = document.getElementById('musicModal');
  let interval;
  const observer = new MutationObserver(() => {
    const visible = musicModal.style.display === 'flex';
    if (visible && !interval) interval = setInterval(refreshStatus, 5000);
    if (!visible && interval) { clearInterval(interval); interval = null; }
  });
  observer.observe(musicModal, { attributes: true, attributeFilter: ['style'] });
}

// Spotify設定の初期化
function initializeSpotifySettings() {
  const clientIdInput = document.getElementById('spotifyClientId');
  const saveBtn = document.getElementById('saveSpotifyClientId');
  const statusText = document.getElementById('spotifyClientStatus');

  // 現在の設定を読み込み
  loadSpotifySettings();

  async function loadSpotifySettings() {
    try {
      const resp = await chrome.runtime.sendMessage({ action: 'SPOTIFY_GET_CLIENT_ID' });
      if (resp && resp.success && resp.clientId) {
        clientIdInput.value = resp.clientId;
        updateStatus(true);
      } else {
        updateStatus(false);
      }
    } catch (e) {
      console.error('Failed to load Spotify client ID:', e);
      updateStatus(false);
    }
  }

  function updateStatus(configured) {
    if (configured) {
      statusText.textContent = '設定済み';
      statusText.className = 'status-text configured';
    } else {
      statusText.textContent = '未設定';
      statusText.className = 'status-text';
    }
  }

  saveBtn?.addEventListener('click', async () => {
    const clientId = clientIdInput.value.trim();
    if (!clientId) {
      alert('クライアントIDを入力してください');
      return;
    }

    saveBtn.disabled = true;
    saveBtn.textContent = '保存中...';

    try {
      const resp = await chrome.runtime.sendMessage({ 
        action: 'SPOTIFY_SET_CLIENT_ID', 
        clientId 
      });
      
      if (resp && resp.success) {
        updateStatus(true);
        // Spotify機能の表示を更新
        if (typeof initializeSpotifyFeatures._refresh === 'function') {
          initializeSpotifyFeatures._refresh();
        }
        alert('クライアントIDが保存されました');
      } else {
        throw new Error(resp?.error || '保存に失敗しました');
      }
    } catch (e) {
      alert(`エラー: ${e.message}`);
    } finally {
      saveBtn.disabled = false;
      saveBtn.textContent = '保存';
    }
  });

  // Enterキーでも保存
  clientIdInput?.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      saveBtn.click();
    }
  });
}

// Spotify設定の初期化
function initializeSpotifySettings() {
  const clientIdInput = document.getElementById('spotifyClientId');
  const saveBtn = document.getElementById('saveSpotifyClientId');
  const statusText = document.getElementById('spotifyClientStatus');

  if (!clientIdInput || !saveBtn || !statusText) return;

  // 現在のクライアントIDを取得して表示
  async function loadCurrentClientId() {
    try {
      const resp = await chrome.runtime.sendMessage({ action: 'SPOTIFY_GET_CLIENT_ID' });
      if (resp && resp.success && resp.clientId) {
        clientIdInput.value = resp.clientId;
        statusText.textContent = '設定済み';
        statusText.className = 'status-text configured';
      } else {
        statusText.textContent = '未設定';
        statusText.className = 'status-text';
      }
    } catch (e) {
      statusText.textContent = '読み込みエラー';
      statusText.className = 'status-text';
    }
  }

  // 保存処理
  saveBtn.addEventListener('click', async () => {
    const clientId = clientIdInput.value.trim();
    if (!clientId) {
      alert('クライアントIDを入力してください');
      return;
    }

    saveBtn.disabled = true;
    saveBtn.textContent = '保存中...';

    try {
      const resp = await chrome.runtime.sendMessage({ 
        action: 'SPOTIFY_SET_CLIENT_ID', 
        clientId 
      });
      
      if (resp && resp.success) {
        statusText.textContent = '設定済み';
        statusText.className = 'status-text configured';
        // Spotify UIを更新
        if (typeof initializeSpotifyFeatures === 'function' && initializeSpotifyFeatures._refresh) {
          initializeSpotifyFeatures._refresh();
        }
      } else {
        alert('保存に失敗しました');
      }
    } catch (e) {
      alert('保存中にエラーが発生しました');
    } finally {
      saveBtn.disabled = false;
      saveBtn.textContent = '保存';
    }
  });

  // Enterキーでも保存
  clientIdInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      saveBtn.click();
    }
  });

  // 初期読み込み
  loadCurrentClientId();
}

// Dynamic IslandからのメッセージをハンドリングしてPopupで設定タブを開く
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'OPEN_SETTINGS_TAB') {
    // 設定タブをクリック
    const settingsTab = document.querySelector('[data-tab="settings"]');
    if (settingsTab) {
      settingsTab.click();
      
      // Spotify設定セクションまでスクロール
      setTimeout(() => {
        scrollToSpotifySettings();
      }, 100);
    }
    sendResponse({ success: true });
  }
  if (message.action === 'OPEN_ACCOUNT_TAB') {
    const accountTab = document.querySelector('[data-tab="account"]');
    if (accountTab) {
      accountTab.click();
    }
    sendResponse({ success: true });
  }
});

// Spotify設定セクションまでスクロールする関数
function scrollToSpotifySettings() {
  const spotifyGroup = document.querySelector('.setting-group:has(.spotify-setting)');
  const spotifyHeader = spotifyGroup?.querySelector('.setting-group-header');
  
  if (spotifyGroup && spotifyHeader) {
    // 設定グループを展開
    if (!spotifyGroup.classList.contains('expanded')) {
      spotifyHeader.click();
    }
    
    // 少し遅延してからスクロール（展開アニメーション完了後）
    setTimeout(() => {
      spotifyGroup.scrollIntoView({ 
        behavior: 'smooth', 
        block: 'start',
        inline: 'nearest'
      });
      
      // さらに入力フィールドにフォーカス
      setTimeout(() => {
        const clientIdInput = document.getElementById('spotifyClientId');
        if (clientIdInput) {
          clientIdInput.focus();
          // フィールドをハイライト
          clientIdInput.select();
        }
      }, 300);
    }, 200);
  }
}
